/*==========================================================================
  MEMREG.C

  MACH 64 functions to access the memory mapped registers.

  Copyright (c) 1993-1995 ATI Technologies Inc. All rights reserved
  =========================================================================*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <conio.h>
#include <dos.h>

#include "atim64.h"
#include "sample.h"

/* --------------------------------------------------------------------------
  APP_REGW - write to a memory mapped register through the linear aperture.

  See REGW () for method description.
-------------------------------------------------------------------------- */
void app_regw (unsigned int regindex, unsigned long regdata)
{
    unsigned long appaddr;
    unsigned long *dataptr;

    // calculate aperture address
    appaddr = modeinfo.aperture_address + modeinfo.linear_memreg_offset +
              regindex;

    // setup dataptr to write from
    dataptr = &regdata;

    movemem (dataptr, appaddr, 2, 1);
}

/* --------------------------------------------------------------------------
  APP_REGR - read from a memory mapped register through the linear aperture.

  See REGR () for method description.
-------------------------------------------------------------------------- */
unsigned long app_regr (unsigned int regindex)
{
    unsigned long appaddr, regdata;
    unsigned long *dataptr;

    // calculate aperture address
    appaddr = modeinfo.aperture_address + modeinfo.linear_memreg_offset +
              regindex;

    // setup dataptr to read to
    dataptr = &regdata;

    movemem (dataptr, appaddr, 2, 0);

    return (regdata);
}


/* --------------------------------------------------------------------------
  REGW - write to a memory mapped register through an available aperture.

  This function will provide write access to the memory mapped registers.
  Each register is 32 bits wide. The appropriate method is selected based
  which aperture is enabled. Preference is given to the small VGA aperture
  since this method will work on all bus types. It is assumed that one of
  the two apertures is enabled.


  VGA aperture method:

    This method will provide write access on all bus types. It is assumed
    that the VGA aperture is enabled. The base address of the memory mapped
    registers is B000:FC00h. Each register occupies 4 bytes. This is also
    the fastest method for real mode applications.


  Linear aperture method:

    This method will provide write access on all bus types except ISA. ISA
    will work if the linear aperture can be enabled without mapping to
    existing extended memory. It is assumed that the linear memory aperture
    is enabled. The base address of the memory mapped registers is (base
    aperture address + 3FFC00h) for a 4M aperture size and (base aperture
    address + 7FFC00h) for an 8M aperture size. Each register occupies 4
    bytes. This method will impose a heavy performance hit since the memory
    mapped registers exist in extended memory above real mode space.
-------------------------------------------------------------------------- */
void regw (unsigned int regindex, unsigned long regdata)
{
    if (modeinfo.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        aregw (regindex, regdata);
    }
    else if (modeinfo.linear_aperture_status == LINEAR_APERTURE_ENABLED)
    {
        app_regw (regindex, regdata);
    }
}

/* --------------------------------------------------------------------------
  REGR - read from a memory mapped register through an available aperture.

  This function will provide read access to the memory mapped registers.
  Each register is 32 bits wide. The appropriate method is selected based
  which aperture is enabled. Preference is given to the small VGA aperture
  since this method will work on all bus types. It is assumed that one of
  the two apertures is enabled.


  VGA aperture method:

    This method will provide read access on all bus types. It is assumed
    that the VGA aperture is enabled. The base address of the memory mapped
    registers is B000:FC00h. Each register occupies 4 bytes. This is also
    the fastest method for real mode applications.


  Linear aperture method:

    This method will provide read access on all bus types except ISA. ISA
    will work if the linear aperture can be enabled without mapping to
    existing extended memory. It is assumed that the linear memory aperture
    is enabled. The base address of the memory mapped registers is (base
    aperture address + 3FFC00h) for a 4M aperture size and (base aperture
    address + 7FFC00h) for an 8M aperture size. Each register occupies 4
    bytes.
-------------------------------------------------------------------------- */
unsigned long regr (unsigned int regindex)
{
    if (modeinfo.vga_aperture_status == VGA_APERTURE_ENABLED)
    {
        return (*((unsigned long far *) (VGA_REGISTER_BASE + regindex)));
    }
    else if (modeinfo.linear_aperture_status == LINEAR_APERTURE_ENABLED)
    {
        return (app_regr (regindex));
    }
}

